# VSCode Setup Guide for Savly Group Savings Platform

## Quick Start Instructions

### 1. Download Project Files
Download these essential files from your Replit project:
- `app.py` - Main Flask application
- `models.py` - Database models
- `routes.py` - Application routes
- `main.py` - Entry point
- `requirements.txt` - Python dependencies
- `templates/` folder - All HTML templates
- `.env.example` - Environment variables template

### 2. VSCode Setup

#### Install Required Extensions:
1. **Python** (Microsoft) - Essential for Python development
2. **Flask-Snippets** - Flask code snippets
3. **SQLTools** - Database management
4. **Python Docstring Generator** - Auto documentation
5. **GitLens** - Enhanced Git integration

#### Recommended Extensions:
- **Prettier** - Code formatting
- **Auto Rename Tag** - HTML tag synchronization
- **Bracket Pair Colorizer** - Better code readability
- **Material Icon Theme** - Better file icons

### 3. Python Environment Setup

```bash
# Create virtual environment
python3 -m venv venv

# Activate virtual environment
# On macOS/Linux:
source venv/bin/activate
# On Windows:
venv\Scripts\activate

# Install dependencies
pip install -r requirements.txt
```

### 4. Database Setup

#### Option A: PostgreSQL (Recommended for Production)
```bash
# Install PostgreSQL (macOS with Homebrew)
brew install postgresql
brew services start postgresql

# Create database
createdb savly_development

# Set environment variable
export DATABASE_URL="postgresql://username:password@localhost/savly_development"
```

#### Option B: SQLite (Quick Development)
```bash
# Simply set in .env file
DATABASE_URL=sqlite:///savly.db
```

### 5. Environment Configuration

Create `.env` file in project root:
```env
# Database
DATABASE_URL=postgresql://username:password@localhost/savly_development
# Or for SQLite: DATABASE_URL=sqlite:///savly.db

# Flask Configuration
FLASK_ENV=development
FLASK_DEBUG=True
SESSION_SECRET=your-super-secret-key-here

# Application Settings
PORT=5001
HOST=0.0.0.0
```

### 6. VSCode Configuration

Create `.vscode/settings.json`:
```json
{
    "python.defaultInterpreterPath": "./venv/bin/python",
    "python.terminal.activateEnvironment": true,
    "python.linting.enabled": true,
    "python.linting.pylintEnabled": true,
    "python.formatting.provider": "black",
    "python.formatting.blackArgs": ["--line-length=88"],
    "emmet.includeLanguages": {
        "jinja-html": "html"
    },
    "files.associations": {
        "*.html": "jinja-html"
    }
}
```

Create `.vscode/launch.json` for debugging:
```json
{
    "version": "0.2.0",
    "configurations": [
        {
            "name": "Flask Debug",
            "type": "python",
            "request": "launch",
            "program": "main.py",
            "console": "integratedTerminal",
            "envFile": "${workspaceFolder}/.env",
            "args": [],
            "jinja": true
        }
    ]
}
```

### 7. Running the Application

#### Method 1: VSCode Debug (Recommended)
1. Press `F5` or click "Run and Debug"
2. Select "Flask Debug" configuration
3. Application will start with debugger attached

#### Method 2: Terminal
```bash
# Make sure virtual environment is activated
source venv/bin/activate

# Run the application
python main.py
```

#### Method 3: Flask CLI
```bash
export FLASK_APP=main.py
export FLASK_ENV=development
flask run --host=0.0.0.0 --port=5001
```

### 8. Database Initialization

```bash
# In Python interactive shell or add to main.py
python
>>> from app import app, db
>>> with app.app_context():
...     db.create_all()
>>> exit()
```

### 9. Testing Access

Open browser and navigate to:
- **Home Page**: http://localhost:5001
- **Admin Login**: http://localhost:5001/admin/login
- **Member Login**: http://localhost:5001/member/login

### 10. Default Login Credentials

#### Superadmin Access:
- Username: `admin`
- Password: `admin123`

#### Group Admin Access:
- Username: `amader`
- Password: `22558822`

#### Member Access:
- Any member (M001-M022)
- Password: `1234`

## Development Workflow

### 1. Git Setup (Recommended)
```bash
git init
git add .
git commit -m "Initial project setup"
```

### 2. Database Migrations
When you modify models.py:
```bash
# Drop and recreate tables (development only)
python
>>> from app import app, db
>>> with app.app_context():
...     db.drop_all()
...     db.create_all()
```

### 3. Adding New Features
1. Create feature branch: `git checkout -b feature/new-feature`
2. Modify code in VSCode
3. Test locally: `python main.py`
4. Commit changes: `git commit -m "Add new feature"`

### 4. Code Quality
```bash
# Install development tools
pip install black pylint flake8

# Format code
black .

# Check code quality
pylint *.py
flake8 .
```

## VSCode Shortcuts

- **F5**: Start debugging
- **Ctrl+Shift+`**: Open terminal
- **Ctrl+Shift+P**: Command palette
- **Ctrl+`**: Toggle terminal
- **F12**: Go to definition
- **Shift+F12**: Find all references

## Troubleshooting

### Port 5000 Conflicts (macOS)
If port 5000 is busy (AirPlay Receiver):
```bash
# Use port 5001 instead (already configured in .env)
python main.py
# Application will run on http://localhost:5001
```

### Database Connection Issues
```bash
# Check PostgreSQL status
brew services list | grep postgresql

# Restart PostgreSQL
brew services restart postgresql

# For SQLite, ensure directory permissions
chmod 755 .
```

### Python Virtual Environment Issues
```bash
# Recreate virtual environment
rm -rf venv
python3 -m venv venv
source venv/bin/activate
pip install -r requirements.txt
```

### Template/Static File Issues
Ensure folder structure:
```
project/
├── app.py
├── models.py
├── routes.py
├── main.py
├── requirements.txt
├── .env
├── templates/
│   ├── admin/
│   ├── member/
│   ├── public/
│   └── superadmin/
└── static/ (optional)
```

## Production Deployment

### For Namecheap cPanel:
1. Upload all files via File Manager
2. Create Python app in cPanel
3. Install requirements: `pip install -r requirements.txt`
4. Set entry point to `main:app`
5. Configure environment variables in cPanel

### For Heroku:
1. Create `Procfile`: `web: gunicorn main:app`
2. Set environment variables in Heroku dashboard
3. Deploy via Git: `git push heroku main`

## Support

If you encounter issues:
1. Check VSCode Python interpreter is set to virtual environment
2. Ensure all dependencies are installed: `pip list`
3. Verify database connection in .env file
4. Check Flask debug output for errors
5. Use VSCode debugger to step through code

## Next Steps

1. Set up the development environment following this guide
2. Test the application locally
3. Make your desired modifications
4. Set up version control with Git
5. Consider deployment options for production use

The complete Savly platform is now ready for VSCode development with full debugging capabilities, proper Python environment management, and professional development workflow.